
'use client';

import { useMemo, useState, useEffect, use } from 'react';
import { useDoc, useFirestore, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import { Product } from '@/lib/data';
import Image from 'next/image';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Star, ThumbsUp } from 'lucide-react';
import { Separator } from '@/components/ui/separator';
import { useSiteConfig } from '@/hooks/use-site-config';
import { Badge } from '@/components/ui/badge';
import { Skeleton } from '@/components/ui/skeleton';
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
  type CarouselApi,
} from "@/components/ui/carousel"
import { cn } from '@/lib/utils';
import { CommentSection } from '@/components/products/CommentSection';

export default function ProductDetailPage({ params }: { params: { productId: string } }) {
  const { productId } = use(params);
  
  const firestore = useFirestore();
  const { siteConfig } = useSiteConfig();
  const productRef = useMemoFirebase(
    () => (firestore ? doc(firestore, 'products', productId) : null),
    [firestore, productId]
  );
  const { data: product, isLoading: isProductLoading } = useDoc<Product>(productRef);

  const [carouselApi, setCarouselApi] = useState<CarouselApi>()
  const [current, setCurrent] = useState(0)

  useEffect(() => {
    if (!carouselApi) {
      return
    }
 
    setCurrent(carouselApi.selectedScrollSnap())
 
    carouselApi.on("select", () => {
      setCurrent(carouselApi.selectedScrollSnap())
    })
  }, [carouselApi])


  const galleryImages = useMemo(() => {
    if (!product) return [];
    const images = [product.image, product.image2, product.image3, product.image4];
    return images.filter((img): img is string => !!img && img.trim() !== '');
  }, [product]);
  
  const whatsappMessage = `Hello, I'm interested in buying this product: ${product?.name}\n\nImage: ${product?.image}`;
  const whatsappUrl = `https://wa.me/232099675032?text=${encodeURIComponent(whatsappMessage)}`;

  const handleThumbnailClick = (index: number) => {
    carouselApi?.scrollTo(index);
  }

  if (isProductLoading) {
    return (
        <div className="container mx-auto px-4 py-8">
            <div className="grid md:grid-cols-2 gap-8">
                <div>
                    <Skeleton className="w-full h-96 rounded-lg" />
                    <div className="flex gap-2 mt-4">
                        <Skeleton className="w-24 h-24 rounded-md" />
                        <Skeleton className="w-24 h-24 rounded-md" />
                        <Skeleton className="w-24 h-24 rounded-md" />
                        <Skeleton className="w-24 h-24 rounded-md" />
                    </div>
                </div>
                <div className="space-y-4">
                    <Skeleton className="h-10 w-3/4" />
                    <Skeleton className="h-6 w-1/4" />
                    <Skeleton className="h-20 w-full" />
                    <Skeleton className="h-12 w-1/2" />
                </div>
            </div>
        </div>
    );
  }

  if (!product) {
    return <div className="container py-12 text-center">Product not found.</div>;
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="grid md:grid-cols-2 gap-8">
        {/* Image Gallery */}
        <div>
           {galleryImages.length > 0 ? (
                <div className="space-y-4">
                    <Carousel setApi={setCarouselApi} className="relative">
                        <CarouselContent>
                            {galleryImages.map((img, index) => (
                                <CarouselItem key={index}>
                                    <div className="aspect-square w-full overflow-hidden rounded-lg border">
                                        <Image
                                            src={img}
                                            alt={`${product.name} image ${index + 1}`}
                                            width={800}
                                            height={800}
                                            className="w-full h-full object-cover"
                                            priority={index === 0}
                                        />
                                    </div>
                                </CarouselItem>
                            ))}
                        </CarouselContent>
                        {galleryImages.length > 1 && (
                            <>
                                <CarouselPrevious className="absolute left-2 top-1/2 -translate-y-1/2" />
                                <CarouselNext className="absolute right-2 top-1/2 -translate-y-1/2" />
                            </>
                        )}
                    </Carousel>
                    {galleryImages.length > 1 && (
                        <div className="flex space-x-2">
                            {galleryImages.map((img, index) => (
                                <button
                                    key={index}
                                    onClick={() => handleThumbnailClick(index)}
                                    className={cn(
                                        'w-24 h-24 rounded-md border-2 overflow-hidden',
                                        current === index ? 'border-primary' : 'border-transparent'
                                    )}
                                >
                                    <Image
                                        src={img}
                                        alt={`${product.name} thumbnail ${index + 1}`}
                                        width={96}
                                        height={96}
                                        className="w-full h-full object-cover"
                                    />
                                </button>
                            ))}
                        </div>
                    )}
                </div>
            ) : (
                <div className="aspect-square w-full overflow-hidden rounded-lg border bg-muted flex items-center justify-center">
                    <p className="text-muted-foreground">No image available</p>
                </div>
            )}
        </div>

        {/* Product Details */}
        <div className="flex flex-col">
            <Badge variant="secondary" className="w-fit">{product.category}</Badge>
            <h1 className="text-3xl md:text-4xl font-extrabold tracking-tight my-2">{product.name}</h1>
            
            <p className="mt-4 text-muted-foreground">{product.description}</p>
            
            <div className="mt-auto pt-8">
                 <Button asChild size="lg" className="w-full">
                    <a href={whatsappUrl} target="_blank" rel="noopener noreferrer">
                        Buy Now
                    </a>
                </Button>
            </div>
        </div>
      </div>

      
      {product.review && (
        <>
            <Separator className="my-12" />
            <div className="max-w-2xl mx-auto">
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2"><ThumbsUp/> Our Take</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div>
                            <div className="flex items-center mb-2">
                                {[...Array(5)].map((_, i) => (
                                    <Star key={i} className={cn("h-5 w-5", (product.rating && i < product.rating) ? "fill-yellow-400 text-yellow-400" : "fill-muted stroke-muted-foreground")} />
                                ))}
                            </div>
                            <blockquote className="border-l-2 pl-4 italic text-muted-foreground">
                            {product.review}
                            </blockquote>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </>
      )}

      <Separator className="my-12" />

      <CommentSection productId={productId} />

    </div>
  );
}
